/**
 * SENTRY CLIENT CONFIGURATION
 * 
 * Tracks frontend errors, performance, and user interactions
 * Automatically captures unhandled errors and promise rejections
 */

import * as Sentry from "@sentry/nextjs";

// Get DSN from environment variable
const SENTRY_DSN = process.env.NEXT_PUBLIC_SENTRY_DSN;
const SENTRY_ENVIRONMENT = process.env.NEXT_PUBLIC_SENTRY_ENVIRONMENT || process.env.NODE_ENV || 'development';

if (SENTRY_DSN) {
  Sentry.init({
    // Required: Your Sentry DSN (Project Settings > Client Keys)
    dsn: SENTRY_DSN,

    // Environment (production, staging, development)
    environment: SENTRY_ENVIRONMENT,

    // Release version (use git commit hash or package version)
    release: process.env.NEXT_PUBLIC_SENTRY_RELEASE || process.env.npm_package_version,

    // Performance Monitoring - Sample rate (0.0 to 1.0)
    // 1.0 = 100% of transactions sent to Sentry
    // Lower in production to reduce quota usage
    tracesSampleRate: SENTRY_ENVIRONMENT === 'production' ? 0.1 : 1.0,

    // Session Replay - Sample rate (0.0 to 1.0)
    // Captures user sessions for debugging
    replaysSessionSampleRate: SENTRY_ENVIRONMENT === 'production' ? 0.1 : 0.5,

    // Session Replay - Sample rate when error occurs
    replaysOnErrorSampleRate: 1.0,

    // Integrations
    integrations: [
      // Session Replay for visual debugging
      Sentry.replayIntegration({
        // Mask all text content (PII protection)
        maskAllText: true,
        // Block all media elements (images, videos)
        blockAllMedia: true,
      }),

      // Browser Tracing for performance monitoring
      Sentry.browserTracingIntegration(),
    ],

    // Before sending events to Sentry
    beforeSend(event, hint) {
      // Filter out specific errors
      const error = hint.originalException;

      if (error && typeof error === 'object' && 'message' in error) {
        const message = String(error.message).toLowerCase();

        // Don't send common browser errors
        if (
          message.includes('network error') ||
          message.includes('failed to fetch') ||
          message.includes('load failed') ||
          message.includes('networkerror')
        ) {
          return null;
        }
      }

      // Scrub sensitive data from event
      if (event.request?.cookies) {
        delete event.request.cookies;
      }

      if (event.request?.headers) {
        delete event.request.headers['authorization'];
        delete event.request.headers['cookie'];
      }

      return event;
    },

    // Ignore specific errors
    ignoreErrors: [
      // Browser extension errors
      'top.GLOBALS',
      'chrome-extension://',
      'moz-extension://',
      // Random network errors
      'NetworkError',
      'Network request failed',
      'Failed to fetch',
      'Load failed',
    ],

    // Disable in development (optional)
    enabled: SENTRY_ENVIRONMENT !== 'development',
  });
} else {
  // Silent fallback - Sentry not initialized without DSN
}

// Export router transition hook for Next.js navigation instrumentation
export const onRouterTransitionStart = Sentry.captureRouterTransitionStart;
