import { NextResponse, type NextRequest } from "next/server";
import { logger } from '@/lib/utils/logger';

// ============= ROUTE DEFINITIONS =============
// Public routes that don't require any authentication
const PUBLIC_ROUTES = [
    '/',
    '/about',
    '/contact',
    '/terms-of-service',
    '/privacy-policy',
    '/installation-guide',
    '/blocked',
    '/e', // Public eSIM page prefix
];

// Auth routes (login, register, etc.)
const AUTH_ROUTES = [
    '/auth',
    '/api/auth',
];

// Public API routes
const PUBLIC_API_ROUTES = [
    '/api/packages',
    '/api/packages/simple',
    '/api/esim', // Public eSIM access
    '/api/csrf-token',
];

// Webhook routes (require signature verification, not user auth)
const WEBHOOK_ROUTES = [
    '/api/webhooks/esimaccess',
    '/api/webhooks/pakasir',
];

// Admin-only routes
const ADMIN_ROUTES = [
    '/admin',
    '/api/admin',
];

// Reseller routes (require reseller or admin role)
const RESELLER_ROUTES = [
    '/dashboard',
    '/api/reseller',
    '/api/orders',
];

// Static/system routes
const STATIC_ROUTES = [
    '/_next',
    '/static',
    '/favicon.ico',
];

// ============= RATE LIMIT CONFIGURATIONS =============
interface RateLimitConfig {
    maxRequests: number;
    windowMs: number;
}

const RATE_LIMITS: Record<string, RateLimitConfig> = {
    // Auth endpoints - strict limits
    'auth_login': { maxRequests: 5, windowMs: 15 * 60 * 1000 },
    'auth_register': { maxRequests: 3, windowMs: 60 * 60 * 1000 },
    'auth_forgot': { maxRequests: 3, windowMs: 60 * 60 * 1000 },
    'auth_reset': { maxRequests: 5, windowMs: 60 * 60 * 1000 },
    // API endpoints - moderate limits
    'api_general': { maxRequests: 100, windowMs: 60 * 1000 },
    'api_admin': { maxRequests: 200, windowMs: 60 * 1000 },
    'api_reseller': { maxRequests: 60, windowMs: 60 * 1000 },
    // Order/financial endpoints - strict limits
    'api_order': { maxRequests: 30, windowMs: 60 * 1000 },
    'api_deposit': { maxRequests: 10, windowMs: 60 * 1000 },
};

// In-memory rate limit store (for edge runtime)
const rateLimitStore = new Map<string, { count: number; resetTime: number }>();

function checkRateLimit(key: string, config: RateLimitConfig): { allowed: boolean; remaining: number } {
    const now = Date.now();
    const record = rateLimitStore.get(key);

    if (!record || now > record.resetTime) {
        rateLimitStore.set(key, { count: 1, resetTime: now + config.windowMs });
        return { allowed: true, remaining: config.maxRequests - 1 };
    }

    if (record.count >= config.maxRequests) {
        return { allowed: false, remaining: 0 };
    }

    record.count++;
    return { allowed: true, remaining: config.maxRequests - record.count };
}

function getClientIP(request: NextRequest): string {
    return request.headers.get('x-forwarded-for')?.split(',')[0].trim() ||
        request.headers.get('x-real-ip') ||
        'unknown';
}

// ============= HELPER FUNCTIONS =============
function matchesRoute(pathname: string, routes: string[]): boolean {
    return routes.some(route => pathname === route || pathname.startsWith(route + '/'));
}

function decodeJWTPayload(token: string): Record<string, unknown> | null {
    try {
        const parts = token.split('.');
        if (parts.length !== 3) return null;
        return JSON.parse(Buffer.from(parts[1], 'base64').toString());
    } catch {
        return null;
    }
}

// ============= MAIN PROXY FUNCTION =============
export async function proxy(request: NextRequest) {
    const { pathname, hostname } = request.nextUrl;
    const clientIP = getClientIP(request);

    // ============= SECURITY HEADERS =============
    const response = NextResponse.next();

    // HTTPS Enforcement in production (skip for localhost/LAN)
    if (
        process.env.NODE_ENV === 'production' &&
        hostname !== 'localhost' &&
        hostname !== '127.0.0.1' &&
        !hostname.startsWith('192.168.') &&
        !hostname.startsWith('10.') &&
        !hostname.startsWith('172.') &&
        request.headers.get('x-forwarded-proto') !== 'https'
    ) {
        const httpsUrl = new URL(request.url);
        httpsUrl.protocol = 'https:';
        return NextResponse.redirect(httpsUrl, 301);
    }

    // HSTS (skip for localhost)
    if (hostname !== 'localhost' && hostname !== '127.0.0.1') {
        response.headers.set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');
    }

    // Content Security Policy
    const isLocalhost = hostname === 'localhost' || hostname === '127.0.0.1' || hostname.startsWith('192.168.');
    const cspPolicy = isLocalhost
        ? "default-src 'self' http: https:; " +
        "script-src 'self' 'unsafe-inline' 'unsafe-eval' cdnjs.cloudflare.com blob:; " +
        "style-src 'self' 'unsafe-inline' fonts.googleapis.com; " +
        "font-src 'self' fonts.gstatic.com cdnjs.cloudflare.com; " +
        "img-src 'self' data: http: https: blob:; " +
        "connect-src 'self' http: https: blob:; " +
        "frame-ancestors 'none';"
        : "default-src 'self'; " +
        "script-src 'self' 'unsafe-inline' 'unsafe-eval' cdnjs.cloudflare.com blob:; " +
        "style-src 'self' 'unsafe-inline' fonts.googleapis.com; " +
        "font-src 'self' fonts.gstatic.com cdnjs.cloudflare.com; " +
        "img-src 'self' data: https: blob:; " +
        "connect-src 'self' https: blob:; " +
        "frame-ancestors 'none';";
    response.headers.set('Content-Security-Policy', cspPolicy);

    // Other security headers
    response.headers.set('X-Content-Type-Options', 'nosniff');
    response.headers.set('X-Frame-Options', 'DENY');
    response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin');
    response.headers.set('Permissions-Policy', 'geolocation=(), microphone=(), camera=(), payment=()');

    // ============= STATIC/SYSTEM ROUTES =============
    if (matchesRoute(pathname, STATIC_ROUTES) || pathname.includes('.')) {
        return NextResponse.next();
    }

    // ============= PUBLIC ROUTES =============
    if (matchesRoute(pathname, PUBLIC_ROUTES)) {
        return response;
    }

    // ============= AUTH ROUTES =============
    if (matchesRoute(pathname, AUTH_ROUTES)) {
        // Apply rate limiting to sensitive auth endpoints
        if (pathname === '/api/auth/login') {
            const rateKey = `auth_login:${clientIP}`;
            const { allowed, remaining } = checkRateLimit(rateKey, RATE_LIMITS['auth_login']);
            if (!allowed) {
                logger.warn('[PROXY] Rate limit exceeded for login', { ip: clientIP.substring(0, 10) + '...' });
                return NextResponse.json(
                    { error: 'Too many login attempts. Please try again later.' },
                    { status: 429, headers: { 'X-RateLimit-Remaining': '0' } }
                );
            }
            response.headers.set('X-RateLimit-Remaining', remaining.toString());
        }
        if (pathname === '/api/auth/register') {
            const rateKey = `auth_register:${clientIP}`;
            const { allowed, remaining } = checkRateLimit(rateKey, RATE_LIMITS['auth_register']);
            if (!allowed) {
                return NextResponse.json(
                    { error: 'Too many registration attempts. Please try again later.' },
                    { status: 429 }
                );
            }
            response.headers.set('X-RateLimit-Remaining', remaining.toString());
        }
        return response;
    }

    // ============= WEBHOOK ROUTES =============
    if (matchesRoute(pathname, WEBHOOK_ROUTES)) {
        // Webhooks have their own signature verification
        return NextResponse.next();
    }

    // ============= PUBLIC API ROUTES =============
    if (matchesRoute(pathname, PUBLIC_API_ROUTES)) {
        // Apply general rate limiting
        const rateKey = `api_general:${clientIP}`;
        const { allowed } = checkRateLimit(rateKey, RATE_LIMITS['api_general']);
        if (!allowed) {
            return NextResponse.json({ error: 'Rate limit exceeded' }, { status: 429 });
        }
        return response;
    }

    // ============= AUTHENTICATED ROUTES =============
    const token = request.cookies.get('auth-token')?.value;

    if (!token) {
        // No token - redirect pages or return 401 for APIs
        if (pathname.startsWith('/api/')) {
            return NextResponse.json({ error: 'Authentication required' }, { status: 401 });
        }
        const loginUrl = new URL('/auth/login', request.url);
        loginUrl.searchParams.set('returnUrl', pathname);
        return NextResponse.redirect(loginUrl);
    }

    // Decode token to get user info
    const payload = decodeJWTPayload(token);
    if (!payload) {
        logger.warn('[PROXY] Invalid token format', { ip: clientIP.substring(0, 10) + '...' });
        if (pathname.startsWith('/api/')) {
            return NextResponse.json({ error: 'Invalid authentication token' }, { status: 401 });
        }
        const loginUrl = new URL('/auth/login', request.url);
        return NextResponse.redirect(loginUrl);
    }

    const userRole = payload.role as string;
    const emailVerified = payload.emailVerified as boolean;
    const userId = payload.userId as string;

    // ============= EMAIL VERIFICATION CHECK =============
    if (!emailVerified && (pathname.startsWith('/dashboard') || pathname.startsWith('/admin'))) {
        return NextResponse.redirect(new URL('/auth/verify-email-pending', request.url));
    }

    // ============= ADMIN ROUTES =============
    if (matchesRoute(pathname, ADMIN_ROUTES)) {
        if (userRole !== 'admin') {
            logger.warn('[PROXY] Unauthorized admin access attempt', {
                userId: userId?.toString().substring(0, 8) + '...',
                role: userRole,
                path: pathname
            });
            if (pathname.startsWith('/api/')) {
                return NextResponse.json({ error: 'Admin access required' }, { status: 403 });
            }
            return NextResponse.redirect(new URL('/dashboard', request.url));
        }

        // Rate limit admin API calls
        if (pathname.startsWith('/api/admin')) {
            const rateKey = `api_admin:${userId}`;
            const { allowed } = checkRateLimit(rateKey, RATE_LIMITS['api_admin']);
            if (!allowed) {
                return NextResponse.json({ error: 'Rate limit exceeded' }, { status: 429 });
            }
        }

        response.headers.set('x-user-id', userId || '');
        response.headers.set('x-user-role', userRole);
        return response;
    }

    // ============= RESELLER ROUTES =============
    if (matchesRoute(pathname, RESELLER_ROUTES)) {
        if (userRole !== 'reseller' && userRole !== 'admin') {
            logger.warn('[PROXY] Unauthorized reseller access attempt', {
                userId: userId?.toString().substring(0, 8) + '...',
                role: userRole,
                path: pathname
            });
            if (pathname.startsWith('/api/')) {
                return NextResponse.json({ error: 'Reseller access required' }, { status: 403 });
            }
            return NextResponse.redirect(new URL('/auth/login', request.url));
        }

        // Rate limit reseller API calls
        if (pathname.startsWith('/api/reseller')) {
            const rateKey = `api_reseller:${userId}`;
            const { allowed } = checkRateLimit(rateKey, RATE_LIMITS['api_reseller']);
            if (!allowed) {
                return NextResponse.json({ error: 'Rate limit exceeded' }, { status: 429 });
            }
        }

        // Stricter rate limits for financial endpoints
        if (pathname.includes('/deposit') || pathname.includes('/order') || pathname.includes('/topup')) {
            const rateKey = `api_order:${userId}`;
            const { allowed } = checkRateLimit(rateKey, RATE_LIMITS['api_order']);
            if (!allowed) {
                return NextResponse.json({ error: 'Too many requests. Please slow down.' }, { status: 429 });
            }
        }

        response.headers.set('x-user-id', userId || '');
        response.headers.set('x-user-role', userRole);
        return response;
    }

    // ============= DEFAULT: REQUIRE AUTHENTICATION =============
    // Any other route not explicitly handled requires authentication
    if (pathname.startsWith('/api/')) {
        // General API rate limiting
        const rateKey = `api_general:${clientIP}`;
        const { allowed } = checkRateLimit(rateKey, RATE_LIMITS['api_general']);
        if (!allowed) {
            return NextResponse.json({ error: 'Rate limit exceeded' }, { status: 429 });
        }
    }

    response.headers.set('x-user-id', userId || '');
    response.headers.set('x-user-role', userRole || '');
    return response;
}

export const config = {
    matcher: [
        "/((?!_next/static|_next/image|favicon.ico|.*\\.(?:svg|png|jpg|jpeg|gif|webp)$).*)",
    ],
};
