require('dotenv').config();
const { Pool } = require('pg');

const pool = new Pool({
    host: process.env.DB_HOST,
    port: parseInt(process.env.DB_PORT || '5432'),
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    ssl: process.env.DB_SSL === 'true' ? { rejectUnauthorized: false } : false
});

async function checkDatabase() {
    const client = await pool.connect();

    try {
        console.log('\n=== Database Connection Info ===');
        console.log('Host:', process.env.DB_HOST);
        console.log('Database:', process.env.DB_NAME);
        console.log('Port:', process.env.DB_PORT || '5432');

        // Get all tables
        console.log('\n=== Tables in Database ===');
        const tablesResult = await client.query(`
      SELECT table_name 
      FROM information_schema.tables 
      WHERE table_schema = 'public' 
      ORDER BY table_name
    `);

        if (tablesResult.rows.length === 0) {
            console.log('No tables found in the database.');
        } else {
            for (const row of tablesResult.rows) {
                console.log(`\n📋 Table: ${row.table_name}`);

                // Get columns for each table
                const columnsResult = await client.query(`
          SELECT column_name, data_type, is_nullable, column_default
          FROM information_schema.columns 
          WHERE table_schema = 'public' AND table_name = $1
          ORDER BY ordinal_position
        `, [row.table_name]);

                for (const col of columnsResult.rows) {
                    console.log(`   - ${col.column_name} (${col.data_type}) ${col.is_nullable === 'NO' ? 'NOT NULL' : ''}`);
                }

                // Get row count
                const countResult = await client.query(`SELECT COUNT(*) as count FROM "${row.table_name}"`);
                console.log(`   Rows: ${countResult.rows[0].count}`);
            }
        }

    } catch (error) {
        console.error('Error:', error.message);
    } finally {
        client.release();
        await pool.end();
    }
}

checkDatabase();
