/**
 * Migration: Add Whitelabel Pakasir & Customer Topup Settings to Resellers
 * 
 * Adds the following columns to the 'resellers' table:
 * - pakasir_slug: Reseller's Pakasir merchant slug
 * - pakasir_api_key: Reseller's Pakasir API key (stored encrypted)
 * - pakasir_enabled: Boolean flag to enable/disable Pakasir integration
 * - allow_customer_topup: Boolean flag to allow customers to topup from esim link
 * - min_balance_for_topup: Minimum reseller balance required for customer topup
 * 
 * Run this migration: node scripts/migrate-pakasir-whitelabel.js
 */

const { Pool } = require('pg');
require('dotenv').config({ path: '.env.local' });

async function migrate() {
    const pool = new Pool({
        host: process.env.DB_HOST,
        port: parseInt(process.env.DB_PORT || '5432'),
        database: process.env.DB_NAME,
        user: process.env.DB_USER,
        password: process.env.DB_PASSWORD,
        ssl: process.env.DB_SSL === 'true' ? { rejectUnauthorized: false } : false,
    });

    const client = await pool.connect();

    try {
        console.log('Starting migration: Add Whitelabel Pakasir columns...');

        // Check if columns already exist
        const checkColumn = await client.query(`
      SELECT column_name 
      FROM information_schema.columns 
      WHERE table_name = 'resellers' AND column_name = 'pakasir_slug'
    `);

        if (checkColumn.rows.length > 0) {
            console.log('Pakasir columns already exist. Skipping migration.');
            return;
        }

        // Add Pakasir whitelabel columns
        await client.query(`
      ALTER TABLE resellers 
      ADD COLUMN IF NOT EXISTS pakasir_slug VARCHAR(100) NULL,
      ADD COLUMN IF NOT EXISTS pakasir_api_key VARCHAR(500) NULL,
      ADD COLUMN IF NOT EXISTS pakasir_enabled BOOLEAN DEFAULT FALSE,
      ADD COLUMN IF NOT EXISTS allow_customer_topup BOOLEAN DEFAULT FALSE,
      ADD COLUMN IF NOT EXISTS min_balance_for_topup DECIMAL(15,2) DEFAULT 100000.00
    `);

        console.log('✅ Added Pakasir whitelabel columns to resellers table');

        // Add source and payment_method to orders table (for tracking customer topups)
        const checkOrderColumn = await client.query(`
      SELECT column_name 
      FROM information_schema.columns 
      WHERE table_name = 'orders' AND column_name = 'source'
    `);

        if (checkOrderColumn.rows.length === 0) {
            await client.query(`
        ALTER TABLE orders 
        ADD COLUMN IF NOT EXISTS source VARCHAR(20) DEFAULT 'panel',
        ADD COLUMN IF NOT EXISTS parent_esim_id INTEGER NULL,
        ADD COLUMN IF NOT EXISTS payment_method VARCHAR(30) DEFAULT 'reseller_balance'
      `);
            console.log('✅ Added source and payment_method columns to orders table');
        } else {
            console.log('Orders columns already exist. Skipping.');
        }

        console.log('');
        console.log('🎉 Migration completed successfully!');
        console.log('');
        console.log('New columns added to resellers:');
        console.log('  - pakasir_slug         : Reseller\'s Pakasir merchant slug');
        console.log('  - pakasir_api_key      : Reseller\'s Pakasir API key');
        console.log('  - pakasir_enabled      : Enable/disable Pakasir integration');
        console.log('  - allow_customer_topup : Allow customers to topup from eSIM link');
        console.log('  - min_balance_for_topup: Minimum balance for customer topup');
        console.log('');
        console.log('New columns added to orders:');
        console.log('  - source         : Where the order came from (panel/store/esim_topup)');
        console.log('  - parent_esim_id : Reference to original eSIM for topups');
        console.log('  - payment_method : How the order was paid (reseller_balance/pakasir_customer)');

    } catch (error) {
        console.error('Migration failed:', error);
        throw error;
    } finally {
        client.release();
        await pool.end();
    }
}

migrate().catch(console.error);
