/**
 * Migration: Add Pricing Tier Settings
 * 
 * Adds the following settings to the 'settings' table:
 * - retail_markup_percent: Markup for End-User Store (roamku.com)
 * 
 * Run this migration: node scripts/migrate-pricing-tiers.js
 */

const { Pool } = require('pg');
require('dotenv').config({ path: '.env.local' });

async function migrate() {
    const pool = new Pool({
        host: process.env.DB_HOST,
        port: parseInt(process.env.DB_PORT || '5432'),
        database: process.env.DB_NAME,
        user: process.env.DB_USER,
        password: process.env.DB_PASSWORD,
        ssl: process.env.DB_SSL === 'true' ? { rejectUnauthorized: false } : false,
    });

    const client = await pool.connect();

    try {
        console.log('Starting migration: Add Pricing Tier Settings...');

        // Add retail_markup_percent to settings (default 20%)
        await client.query(`
      INSERT INTO settings (key, value, updated_at)
      VALUES ('retail_markup_percent', '20', NOW())
      ON CONFLICT (key) DO NOTHING
    `);
        console.log('✅ Added retail_markup_percent setting (default 20%)');

        // Ensure tiers table has all required columns
        await client.query(`
      ALTER TABLE tiers 
      ADD COLUMN IF NOT EXISTS min_deposit_amount DECIMAL(15,2) DEFAULT 0,
      ADD COLUMN IF NOT EXISTS max_deposit_amount DECIMAL(15,2) DEFAULT 999999999.99,
      ADD COLUMN IF NOT EXISTS description TEXT
    `);
        console.log('✅ Ensured tiers table has deposit range columns');

        // Check if default tiers exist, create them if not
        const tiersCheck = await client.query('SELECT COUNT(*) as count FROM tiers');
        if (parseInt(tiersCheck.rows[0].count) === 0) {
            console.log('Creating default pricing tiers...');
            await client.query(`
        INSERT INTO tiers (name_tiers, markup_percentage, min_deposit_amount, max_deposit_amount, description, is_active, created_at, updated_at)
        VALUES 
          ('Bronze', 8, 0, 1000000, 'Default tier for new resellers', true, NOW(), NOW()),
          ('Silver', 6, 1000001, 5000000, 'Discount tier for active resellers', true, NOW(), NOW()),
          ('Gold', 4, 5000001, 999999999.99, 'Premium tier for top resellers', true, NOW(), NOW())
        ON CONFLICT DO NOTHING
      `);
            console.log('✅ Created default tiers (Bronze 8%, Silver 6%, Gold 4%)');
        } else {
            console.log('Tiers already exist, skipping default tier creation.');
        }

        console.log('');
        console.log('🎉 Migration completed successfully!');
        console.log('');
        console.log('Pricing Tier Hierarchy:');
        console.log('  Provider Price → Base cost from API');
        console.log('  Reseller Price → Provider + Tier Markup (Bronze 8%, Silver 6%, Gold 4%)');
        console.log('  End-User Store → Reseller Price + Retail Markup (default 20%)');
        console.log('');
        console.log('Settings added:');
        console.log('  - retail_markup_percent: Markup applied on roamku.com store (default 20%)');

    } catch (error) {
        console.error('Migration failed:', error);
        throw error;
    } finally {
        client.release();
        await pool.end();
    }
}

migrate().catch(console.error);
