#!/usr/bin/env python3
"""
Pakasir Payment Simulation Script
Untuk testing webhook di mode Sandbox
"""

import requests
import json
import sys

# Configuration - sesuaikan dengan project kamu
CONFIG = {
    "project": "depodomain",      # Ganti dengan project name kamu
    "api_key": "xxx123",          # Ganti dengan API key kamu
    "base_url": "https://app.pakasir.com/api/paymentsimulation"
}

def simulate_payment(order_id: str, amount: int):
    """
    Simulate payment untuk order tertentu
    
    Args:
        order_id: ID order/invoice yang mau disimulasi pembayarannya
        amount: Jumlah pembayaran dalam Rupiah
    """
    payload = {
        "project": CONFIG["project"],
        "order_id": order_id,
        "amount": amount,
        "api_key": CONFIG["api_key"]
    }
    
    print(f"\n🔄 Simulating payment...")
    print(f"   Order ID: {order_id}")
    print(f"   Amount: Rp {amount:,}")
    print(f"   Project: {CONFIG['project']}")
    print("-" * 40)
    
    try:
        response = requests.post(
            CONFIG["base_url"],
            headers={"Content-Type": "application/json"},
            json=payload,
            timeout=30
        )
        
        print(f"\n📡 Response Status: {response.status_code}")
        
        try:
            result = response.json()
            print(f"📦 Response Body:")
            print(json.dumps(result, indent=2, ensure_ascii=False))
        except json.JSONDecodeError:
            print(f"📦 Response Text: {response.text}")
        
        if response.status_code == 200:
            print("\n✅ Payment simulation successful!")
        else:
            print(f"\n❌ Payment simulation failed (HTTP {response.status_code})")
            
        return response
        
    except requests.exceptions.RequestException as e:
        print(f"\n❌ Error: {e}")
        return None


def main():
    if len(sys.argv) < 3:
        print("Usage: python pakasir_simulate.py <order_id> <amount>")
        print("Example: python pakasir_simulate.py INV123123 99000")
        print("\n⚠️  Pastikan kamu sudah update CONFIG di script ini:")
        print("   - project: nama project pakasir kamu")
        print("   - api_key: API key pakasir kamu")
        sys.exit(1)
    
    order_id = sys.argv[1]
    try:
        amount = int(sys.argv[2])
    except ValueError:
        print("❌ Error: amount harus angka")
        sys.exit(1)
    
    simulate_payment(order_id, amount)


if __name__ == "__main__":
    main()
