/**
 * Email Test Script
 * Run with: npx tsx scripts/test-email.ts [email]
 * 
 * Example: npx tsx scripts/test-email.ts test@gmail.com
 */

// Load environment variables FIRST before any other imports
import { config } from 'dotenv'
config()

// Now dynamically import the mailer after env is loaded
async function main() {
    const testEmail = process.argv[2]

    if (!testEmail) {
        console.log('Usage: npx tsx scripts/test-email.ts <email-address>')
        console.log('Example: npx tsx scripts/test-email.ts yourname@gmail.com')
        process.exit(1)
    }

    // Dynamic import AFTER dotenv is configured
    const { verifySmtpConnection, sendTestEmail } = await import('../lib/email/mailer')

    console.log('\n========================================')
    console.log('🔧 EMAIL DIAGNOSTIC TEST')
    console.log('========================================\n')

    // Debug: Show loaded config
    console.log('📋 Environment Config:')
    console.log(`  SMTP_HOST: ${process.env.SMTP_HOST || '(default: mail.roamku.com)'}`)
    console.log(`  SMTP_PORT: ${process.env.SMTP_PORT || '(default: 465)'}`)
    console.log(`  SMTP_PASSWORD: ${process.env.SMTP_PASSWORD ? '***SET***' : '❌ NOT SET'}`)
    console.log('')

    // Step 1: Verify SMTP Connection
    console.log('📡 Step 1: Verifying SMTP Connection...\n')

    const accounts = ['NOTIFICATIONS', 'VERIFY', 'SECURITY', 'BILLING', 'SUPPORT'] as const

    for (const account of accounts) {
        const result = await verifySmtpConnection(account)
        if (result.success) {
            console.log(`  ✅ ${account}: Connected (${result.email})`)
        } else {
            console.log(`  ❌ ${account}: FAILED - ${result.error}`)
            if (result.details) {
                console.log(`     Code: ${result.details.code}`)
                console.log(`     Response: ${result.details.response}`)
            }
        }
    }

    // Step 2: Send Test Email
    console.log('\n📧 Step 2: Sending Test Email...\n')
    console.log(`  Target: ${testEmail}`)
    console.log(`  Account: NOTIFICATIONS\n`)

    const emailResult = await sendTestEmail(testEmail, 'NOTIFICATIONS')

    if (emailResult.success) {
        console.log('  ✅ Email sent successfully!')
        console.log(`  Message ID: ${emailResult.messageId}`)
        console.log(`  Accepted: ${emailResult.accepted?.join(', ') || 'N/A'}`)
        console.log(`  Rejected: ${emailResult.rejected?.join(', ') || 'None'}`)
        console.log(`  SMTP Response: ${emailResult.response}`)
        console.log(`  Envelope: ${JSON.stringify(emailResult.envelope)}`)
    } else {
        console.log('  ❌ Email failed to send!')
        console.log(`  Error: ${emailResult.error}`)
        if (emailResult.errorDetails) {
            console.log(`  Code: ${emailResult.errorDetails.code}`)
            console.log(`  Command: ${emailResult.errorDetails.command}`)
            console.log(`  Response: ${emailResult.errorDetails.response}`)
        }
    }

    console.log('\n========================================')
    console.log('📋 NEXT STEPS')
    console.log('========================================')

    if (emailResult.success) {
        console.log(`
1. Check inbox of ${testEmail}
2. If not in inbox, CHECK SPAM FOLDER
3. If not in spam either, check DNS records:
   - SPF: https://mxtoolbox.com/spf.aspx?domain=roamku.com
   - DKIM: Check with mail server admin
   - DMARC: https://mxtoolbox.com/dmarc.aspx?domain=roamku.com
`)
    } else {
        console.log(`
1. Verify SMTP_PASSWORD is set correctly in .env
2. Check if mail.roamku.com:465 is accessible
3. Try: telnet mail.roamku.com 465
4. Contact mail server administrator
`)
    }

    process.exit(emailResult.success ? 0 : 1)
}

main().catch(console.error)
