/**
 * SENTRY SERVER CONFIGURATION
 * 
 * Tracks backend errors, API failures, and database issues
 * Automatically captures unhandled errors and promise rejections
 */

import * as Sentry from "@sentry/nextjs";

// Get DSN from environment variable
const SENTRY_DSN = process.env.SENTRY_DSN;
const SENTRY_ENVIRONMENT = process.env.SENTRY_ENVIRONMENT || process.env.NODE_ENV || 'development';

if (SENTRY_DSN) {
  Sentry.init({
    // Required: Your Sentry DSN (Project Settings > Client Keys)
    dsn: SENTRY_DSN,

    // Environment (production, staging, development)
    environment: SENTRY_ENVIRONMENT,

    // Release version (use git commit hash or package version)
    release: process.env.SENTRY_RELEASE || process.env.npm_package_version,

    // Performance Monitoring - Sample rate (0.0 to 1.0)
    // 1.0 = 100% of transactions sent to Sentry
    // Lower in production to reduce quota usage
    tracesSampleRate: SENTRY_ENVIRONMENT === 'production' ? 0.1 : 1.0,

    // Before sending events to Sentry
    beforeSend(event, hint) {
      // Scrub sensitive data from event

      // Remove passwords from request data
      if (event.request?.data) {
        const data = event.request.data as Record<string, unknown>;
        if ('password' in data) data.password = '[REDACTED]';
        if ('currentPassword' in data) data.currentPassword = '[REDACTED]';
        if ('newPassword' in data) data.newPassword = '[REDACTED]';
      }

      // Remove cookies and auth headers
      if (event.request?.cookies) {
        delete event.request.cookies;
      }

      if (event.request?.headers) {
        delete event.request.headers['authorization'];
        delete event.request.headers['cookie'];
        delete event.request.headers['x-csrf-token'];
      }

      // Remove environment variables from context
      if (event.contexts?.runtime?.environment) {
        delete event.contexts.runtime.environment;
      }

      return event;
    },

    // Ignore specific errors
    ignoreErrors: [
      // Database connection errors (handled by connection pool)
      'ECONNREFUSED',
      'ENOTFOUND',
      // Expected auth errors
      'Unauthorized',
      'Invalid token',
      'Token expired',
    ],

    // Disable in development (optional)
    enabled: SENTRY_ENVIRONMENT !== 'development',
  });
} else {
  // Silent fallback - Sentry not initialized without DSN
}
