/**
 * Custom Server for Shared Hosting (cPanel)
 * 
 * This server provides:
 * - Graceful shutdown handling (SIGTERM, SIGINT)
 * - Request timeout handling
 * - Health check endpoint
 * - Improved error logging
 * - Process memory monitoring
 * 
 * Usage: node server.js
 * Or via PM2: pm2 start ecosystem.config.js
 */

const { createServer } = require('http');
const { parse } = require('url');
const next = require('next');

// Configuration
const dev = process.env.NODE_ENV !== 'production';
const hostname = process.env.HOST || '0.0.0.0';
const port = parseInt(process.env.PORT || '3000', 10);
const requestTimeout = parseInt(process.env.REQUEST_TIMEOUT || '30000', 10); // 30 seconds default

// Initialize Next.js app
const app = next({ dev, hostname, port });
const handle = app.getRequestHandler();

// Server instance (for graceful shutdown)
let server = null;
let isShuttingDown = false;

// Startup timestamp
const startTime = Date.now();

/**
 * Format bytes to human readable string
 */
function formatBytes(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(2) + ' KB';
    return (bytes / (1024 * 1024)).toFixed(2) + ' MB';
}

/**
 * Get process memory usage
 */
function getMemoryUsage() {
    const usage = process.memoryUsage();
    return {
        heapUsed: formatBytes(usage.heapUsed),
        heapTotal: formatBytes(usage.heapTotal),
        rss: formatBytes(usage.rss),
        external: formatBytes(usage.external),
    };
}

/**
 * Health check response
 */
function handleHealthCheck(res) {
    const uptime = Math.floor((Date.now() - startTime) / 1000);
    const memory = getMemoryUsage();

    const health = {
        status: 'healthy',
        timestamp: new Date().toISOString(),
        uptime: `${uptime} seconds`,
        environment: process.env.NODE_ENV || 'development',
        nodeVersion: process.version,
        memory: memory,
        pid: process.pid,
    };

    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify(health, null, 2));
}

/**
 * Graceful shutdown handler
 */
function gracefulShutdown(signal) {
    if (isShuttingDown) {
        console.log(`[SERVER] Already shutting down, ignoring ${signal}`);
        return;
    }

    isShuttingDown = true;
    console.log(`\n[SERVER] Received ${signal}. Graceful shutdown starting...`);

    // Stop accepting new connections
    if (server) {
        server.close((err) => {
            if (err) {
                console.error('[SERVER] Error during server close:', err);
                process.exit(1);
            }
            console.log('[SERVER] HTTP server closed');

            // Close Next.js app
            app.close().then(() => {
                console.log('[SERVER] Next.js app closed');
                console.log('[SERVER] Graceful shutdown complete');
                process.exit(0);
            }).catch((closeErr) => {
                console.error('[SERVER] Error closing Next.js app:', closeErr);
                process.exit(1);
            });
        });

        // Force shutdown after 30 seconds
        setTimeout(() => {
            console.error('[SERVER] Forced shutdown after timeout');
            process.exit(1);
        }, 30000);
    } else {
        process.exit(0);
    }
}

// Prepare and start the server
app.prepare().then(() => {
    server = createServer(async (req, res) => {
        // Skip if shutting down
        if (isShuttingDown) {
            res.writeHead(503, { 'Content-Type': 'application/json' });
            res.end(JSON.stringify({ error: 'Server is shutting down' }));
            return;
        }

        // Set request timeout
        req.setTimeout(requestTimeout);
        res.setTimeout(requestTimeout);

        // Handle timeout
        req.on('timeout', () => {
            console.error(`[SERVER] Request timeout: ${req.url}`);
            if (!res.headersSent) {
                res.writeHead(408, { 'Content-Type': 'application/json' });
                res.end(JSON.stringify({ error: 'Request timeout' }));
            }
        });

        try {
            const parsedUrl = parse(req.url, true);
            const { pathname } = parsedUrl;

            // Health check endpoint
            if (pathname === '/api/health' || pathname === '/_health') {
                return handleHealthCheck(res);
            }

            // Handle all other requests with Next.js
            await handle(req, res, parsedUrl);

        } catch (err) {
            console.error('[SERVER] Error handling request:', req.url);
            console.error('[SERVER] Error details:', err.message || err);

            // Only send error response if headers haven't been sent
            if (!res.headersSent) {
                res.writeHead(500, { 'Content-Type': 'application/json' });
                res.end(JSON.stringify({
                    error: 'Internal Server Error',
                    message: dev ? err.message : 'An unexpected error occurred'
                }));
            }
        }
    });

    // Handle server errors
    server.on('error', (err) => {
        if (err.code === 'EADDRINUSE') {
            console.error(`[SERVER] Port ${port} is already in use`);
            process.exit(1);
        }
        console.error('[SERVER] Server error:', err);
    });

    // Start listening
    server.listen(port, hostname, () => {
        console.log('');
        console.log('╔════════════════════════════════════════════════════════════╗');
        console.log('║                    RoamKU eSIM Platform                    ║');
        console.log('╠════════════════════════════════════════════════════════════╣');
        console.log(`║  Status:      Ready                                        ║`);
        console.log(`║  URL:         http://${hostname}:${port}                          ║`);
        console.log(`║  Environment: ${(process.env.NODE_ENV || 'development').padEnd(43)}║`);
        console.log(`║  Node:        ${process.version.padEnd(43)}║`);
        console.log(`║  PID:         ${String(process.pid).padEnd(43)}║`);
        console.log('╠════════════════════════════════════════════════════════════╣');
        console.log('║  Health Check: /api/health                                 ║');
        console.log('╚════════════════════════════════════════════════════════════╝');
        console.log('');

        // Log memory usage on startup
        const memory = getMemoryUsage();
        console.log(`[SERVER] Memory: Heap ${memory.heapUsed}/${memory.heapTotal}, RSS ${memory.rss}`);
    });

    // Register shutdown handlers
    process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
    process.on('SIGINT', () => gracefulShutdown('SIGINT'));

    // Handle uncaught exceptions
    process.on('uncaughtException', (err) => {
        console.error('[SERVER] Uncaught Exception:', err);
        gracefulShutdown('uncaughtException');
    });

    // Handle unhandled promise rejections
    process.on('unhandledRejection', (reason, promise) => {
        console.error('[SERVER] Unhandled Rejection at:', promise, 'reason:', reason);
    });

}).catch((err) => {
    console.error('[SERVER] Failed to start:', err);
    process.exit(1);
});
